/*
 * CameraAdjustment.h
 *
 * Created 8/6/2009 By Johnny Huynh
 *
 * Version 00.00.01 8/6/2009
 *
 * Copyright Information:
 * All content copyright  2009 Johnny Huynh. All rights reserved.
 */
 
 #ifndef CAMERA_AJUSTMENT_H
 #define CAMERA_AJUSTMENT_H
 
 #include "global.h"
 
 namespace CameraAdjustment
 {
    template <typename T> inline void adjust_camera_to_first_person_character( NodePath& cam, const NodePath& character, 
                                                                               const T& distance_offset );
    template <typename T> inline void adjust_camera_to_first_person_character( NodePath& cam, const NodePath& character, 
                                                                               const T& distance_offset, 
                                                                               const T& height_offset );
    template <typename T> inline void adjust_camera_to_third_person_character( NodePath& cam, const NodePath& character, 
                                                                               const T& distance_offset );
    template <typename T> inline void adjust_camera_to_third_person_character( NodePath& cam, const NodePath& character, 
                                                                               const T& distance_offset, 
                                                                               const T& height_offset );
    template <typename T> inline void adjust_camera_heading_gradually( NodePath& cam, T heading_degrees,
                                                       const T& readjustment_ratio, const T& min_readjustment_degrees );
 }
 
 /**
  * adjust_camera_to_first_person_character() adjusts the specified camera to look at the specified 
  * actor from the specified distance offset (on the XY-plane) behind the actor.
  *
  * @param (NodePath&) cam
  * @param (const NodePath&) actor
  * @param (const T&) distance_offset
  */
 template <typename T> 
 inline void CameraAdjustment::adjust_camera_to_first_person_character( NodePath& cam, const NodePath& actor, 
                                                                     const T& distance_offset )
 {
    // actor_dir is a unit vector
    VECTOR2_TYPE actor_dir( Vector::get_xy_direction_normalized( actor.get_h() ) );
    
    cam.set_x( actor.get_x() - ( distance_offset * actor_dir.get_x() ) );
    cam.set_y( actor.get_y() - ( distance_offset * actor_dir.get_y() ) );
    cam.look_at( actor, ZERO, ZERO, ZERO );
 }
 
 /**
  * adjust_camera_to_first_person_character() adjusts the specified camera to look at the specified 
  * actor from the specified distance offset (on the XY-plane) behind the actor and the specified height
  * offset above the actor.
  *
  * @param (NodePath&) cam
  * @param (const NodePath&) actor
  * @param (const T&) distance_offset
  * @param (const T&) height_offset
  */
 template <typename T> 
 inline void CameraAdjustment::adjust_camera_to_first_person_character( NodePath& cam, const NodePath& actor, 
                                                                     const T& distance_offset, const T& height_offset )
 {
    T height( height_offset + actor.get_z() );
    if ( cam.get_z() != height )
        cam.set_z( height );
    
    CameraAdjustment::adjust_camera_to_first_person_character( cam, actor, distance_offset );
 }
 
 /**
  * adjust_camera_to_third_person_character() adjusts the specified camera to look at the 
  * specified actor from the specified distance offset (on the XY-plane) away from the actor,
  * without changing the heading angle of the camera (i.e. theoretically, cam.get_h() should
  * not change after this function is called).
  *
  * @param (NodePath&) cam
  * @param (const NodePath&) actor
  * @param (const T&) distance_offset
  */
 template <typename T> 
 inline void CameraAdjustment::adjust_camera_to_third_person_character( NodePath& cam, const NodePath& actor, 
                                                                     const T& distance_offset )
 {
    // cam_dir is a unit vector
    VECTOR2_TYPE cam_dir( Vector::get_xy_direction_normalized( cam.get_h() ) );
    
    cam.set_x( actor.get_x() - ( distance_offset * cam_dir.get_x() ) );
    cam.set_y( actor.get_y() - ( distance_offset * cam_dir.get_y() ) );
    
    cam.look_at( actor, ZERO, ZERO, ZERO );
 }
 
 /**
  * adjust_camera_to_third_person_character() adjusts the specified camera to look at the 
  * specified actor from the specified distance offset (on the XY-plane) away from the actor,
  * without changing the heading angle of the camera (i.e. theoretically, cam.get_h() should
  * not change after this function is called). The height of the camera is also set to be
  * the specified height offset above the actor.
  *
  * @param (NodePath&) cam
  * @param (const NodePath&) actor
  * @param (const T&) distance_offset
  * @param (const T&) height_offset
  */
 template <typename T> 
 inline void CameraAdjustment::adjust_camera_to_third_person_character( NodePath& cam, const NodePath& actor, 
                                                                     const T& distance_offset, const T& height_offset )
 {
    T height( height_offset + actor.get_z() );
    if ( cam.get_z() != height )
        cam.set_z( height );
    
    CameraAdjustment::adjust_camera_to_third_person_character( cam, actor, distance_offset );
 }
 
 /**
  * adjust_camera_heading_gradually() gradually rotates the heading angle of the specified camera 
  * towards facing in the direction of the specified heading angle in degrees. The rotation rate
  * is specified by the readjustment_ratio, which denotes a fraction for the difference between the
  * heading angle of the specified camera and the specified heading angle in degrees. The 
  * min_readjustment_degrees determines the rotation rate if the value calculated from the 
  * readjustment_ratio is less than the min_readjustment_degrees.
  *
  * @param (NodePath&) cam
  * @param (T) heading_degrees
  * @param (const T&) readjustment_ratio - this value should be positive (i.e. greater than zero, but no more than one)
  * @param (const T&) min_reajustment_degrees - this value must be non-negative
  */
 template <typename T>
 inline void CameraAdjustment::adjust_camera_heading_gradually( NodePath& cam, T heading_degrees,
                                        const T& readjustment_ratio, const T& min_readjustment_degrees )
 {
    if ( heading_degrees < ZERO || heading_degrees >= 360.0f )
        heading_degrees = Math3D::get_readjusted_angle_degrees( heading_degrees );
    
    // Slowly readjust the camera's heading angle to face the target heading angle
    T camera_angle( Math3D::get_readjusted_angle_degrees( cam.get_h() ) );
    
    if ( camera_angle != heading_degrees )
    {
        T angle_diff = heading_degrees - camera_angle; // difference in degree
        
        if ( angle_diff > ZERO )
        {
            if ( angle_diff > 180.0f )
            {
                angle_diff -= 360.0f; // degree will be negative after this
                //angle_diff *= THIRD_PERSON_VIEW_READJUSTMENT_RATIO;
                
                //cam.set_h( camera_angle + angle_diff );
            }
            //else // camera_degree <= 180.0f
            //{
                //angle_diff *= THIRD_PERSON_VIEW_READJUSTMENT_RATIO;
                //cam.set_h( camera_angle + angle_diff );
            //}
        }
        else // angle_diff < ZERO
        {
            if ( angle_diff < -180.0f )
            {
                angle_diff += 360.0f; // degree will be positive after this
                //angle_diff *= THIRD_PERSON_VIEW_READJUSTMENT_RATIO;
                
                //cam.set_h( camera_angle + angle_diff );
            }
            //else // camera_degree >= -180.0f
            //{
                //angle_diff *= THIRD_PERSON_VIEW_READJUSTMENT_RATIO;
                //cam.set_h( camera_angle + angle_diff );
            //}
        }
        
        angle_diff *= readjustment_ratio;
        // readjust the camera by at least the degrees of THIRD_PERSON_VIEW_MIN_READJUSTMENT
        if ( angle_diff > ZERO )
        {
            angle_diff = MAX( min_readjustment_degrees, angle_diff );
        }
        else // angle_diff < ZERO
        {
            angle_diff = MIN( -min_readjustment_degrees, angle_diff );
        }
        
        // if readjusting the camera's orientation would move the camera pass the target angle
        if ( (camera_angle < heading_degrees && camera_angle + angle_diff > heading_degrees)
            || (camera_angle > heading_degrees && camera_angle + angle_diff < heading_degrees) )
        {
            // set the camera's heading angle to the targeted heading angle
            cam.set_h( heading_degrees );
        }
        else
        {
            cam.set_h( camera_angle + angle_diff );
        }
    }
 }
 
 #endif // CAMERA_AJUSTMENT_H